<?php

namespace App\Http\Controllers;

use App\Http\Controllers\Controller;
use App\Scene;
use Illuminate\Http\Request;
use Illuminate\Support\Str;
use Illuminate\Support\Facades\Storage;
use App\Config;

class SceneController extends Controller
{

    public function create(Request $request) {
        // create scene
        $scene = new Scene();
        $scene->code = Str::random(10);
        $scene->user_id = $request->user()->id;
        $scene->save();

        return redirect()->action('SceneController@edit', ['id' => $scene->id]);
    }



    public function index(Request $request) {

        $scenes = Scene::with('entities', 'marker')
                        ->where('user_id', $request->user()->id)
                        ->orderBy('created_at', 'desc')
                        ->paginate($this->itens_page);
      
        foreach ($scenes as $scene) {
            //Marker
            $scene->marker = $scene->marker?? "";
            if(!empty($scene->marker)) {
                $scene->marker->props = json_decode($scene->marker->props);
            }
            // Entities
            $scene->entities = ($scene->entities)->map(function($entity) {
                $entity->props = json_decode($entity->props);
                return $entity;
            });
        }
        if($request->query('type') == 'json')
            return response()->json(['response' => 'OK', 'scenes' => $scenes]);
        return view('scene.index', ['scenes' => $scenes]);
    }



    public function admin_index(Request $request) {

        $scenes = Scene::with('entities', 'marker')
                        ->orderBy('created_at', 'desc')
                        ->paginate($this->itens_page);
        
        foreach ($scenes as $scene) {
            //Marker
            $scene->marker = $scene->marker?? "";
            if(!empty($scene->marker)) {
                $scene->marker->props = json_decode($scene->marker->props);
            }
            // Entities
            $scene->entities = ($scene->entities)->map(function($entity) {
                $entity->props = json_decode($entity->props);
                return $entity;
            });
        }
        if($request->query('type') == 'json')
            return response()->json(['response' => 'OK', 'scenes' => $scenes]);
        return view('scene.admin_index', ['scenes' => $scenes]);
    }




    public function view(Scene $scene)
    {
        $this->authorize('view', $scene);

        // Marker
        $scene->marker = $scene->marker?? "";
        if(!empty($scene->marker)) {
            $scene->marker->props = json_decode($scene->marker->props);
            $scene->marker->file = Storage::url("markers/{$scene->marker->file}");
        }
        // Entities
        $scene->entities = ($scene->entities)->map(function($entity) {
            $entity->props = json_decode($entity->props);
            return $entity;
        }); 
        return response()->json(['response' => 'OK', 'scene' => $scene]);
    }


    public function edit(Scene $scene)
    {
        $this->authorize('_edit', $scene);

        // Marker
        $scene->marker = $scene->marker?? "";
        if(!empty($scene->marker)) {
            $scene->marker->props = json_decode($scene->marker->props);
        }
        // Entities
        $scene->entities = ($scene->entities)->map(function($entity) {
            $entity->props = json_decode($entity->props);
            return $entity;
        });
        // Configs
        $configs['APP_QRCODE_CAPTION_ENABLED'] = Config::opt('APP_QRCODE_CAPTION_ENABLED');
        $configs['APP_QRCODE_CAPTION_EDITABLE'] = Config::opt('APP_QRCODE_CAPTION_EDITABLE'); 
        $configs['APP_QRCODE_CAPTION_TEXT'] = Config::opt('APP_QRCODE_CAPTION_TEXT'); 
        $configs['APP_QRCODE_CAPTION_TEXT_FONTSIZE'] = Config::opt('APP_QRCODE_CAPTION_TEXT_FONTSIZE'); 
        $configs['APP_UPLOAD_MAX_FILESIZE'] = Config::opt('APP_UPLOAD_MAX_FILESIZE'); 
        $configs['APP_UPLOAD_MAX_MEDIA_WIDTH'] = Config::opt('APP_UPLOAD_MAX_MEDIA_WIDTH'); 
        $configs['APP_UPLOAD_MAX_MEDIA_HEIGHT'] = Config::opt('APP_UPLOAD_MAX_MEDIA_HEIGHT'); 
        $configs['APP_UPLOAD_MAX_MEDIA_DURATION'] = Config::opt('APP_UPLOAD_MAX_MEDIA_DURATION');

        return view('scene.edit', ['scene' => $scene, 'configs' => $configs]);
    }



    public function addOrEdit(Request $request, Scene $scene=null)
    {

        if($request->isMethod('put')) {
            $this->authorize('edit', $scene);
        } else if($request->isMethod('post')) {
            $this->authorize('add', Scene::class);
            $scene = new Scene();
            $scene->code = Str::random(10);
            $scene->user_id = $request->user()->id;
        }

        $validation = [
            'title' => 'present|nullable|string|max:255',
            'description' => 'present|nullable|string',
            'preload_assets' => 'required|numeric|in:0,1',          // 0 - no, 1 - yes
            'status' => 'required|numeric|in:0,1,2',               // 0 - draft, 1 - published, 2 - archived
            'editable' => 'sometimes|numeric|in:0,1'               // 0 - no, 1 - yes
        ];

        // Publishing
        if($request->status == 1) {
            $validation['title'] = 'required|string|max:255';
            $validation['description'] = 'required|string';
            $validation['published_at'] = 'required|date_format:Y-m-d H:i:s';
        }

        $request->validate($validation);

        $scene->title = $request->title;
        $scene->description = $request->description;
        $scene->preload_assets = $request->preload_assets;

        // Only admin can archive and unarchive scene
        // make the scene editable or non-editable
        if($request->user()->role == 'admin') {
            if($request->status == 1 && $scene->status != 1) {
                $scene->published_at = $request->published_at;
            }
            $scene->status = $request->status;
            $scene->editable = $request->editable;
        }

        if($request->user()->role == 'editor') {
            if($request->status != 2) {
                if($request->status == 1 && $scene->status != 1) {
                    $scene->published_at = $request->published_at;
                }
                $scene->status = $request->status;
            }
        }

        $scene->save();
        return response()->json(['response' => 'OK', 'scene' => $scene]);
    }


   
    public function delete(Request $request, Scene $scene)
    {
        $this->authorize('delete', $scene);

        // Delete Entities
        $entities = $scene->entities;
        if(!empty($entities)) {
            foreach ($entities as $entity) {
                $entity->props = json_decode($entity->props);
                $file = $entity->props->asset->file?? false;
                if ($file) {
                    Storage::delete("assets/$file");
                }
                $entity->delete();
            }
        }

        // Delete Marker
        $marker = $scene->marker?? false;
        if ($marker) {
            $file = $marker->file?? false;
            if ($file) {
                Storage::delete("markers/$file");
            }
            $marker->delete();
        }
        // Delete Scene
        $scene->delete();

        
        if ($request->query('type') == 'json') {
            return response()->json(['response' => 'OK', 'message' => 'deleted']);
        } else {
            $request->session()->flash('status', __('Scene deleted.'));
            $action = $request->query('action', 'index');
            return redirect()->action('SceneController@' . $action);
        }
        
    }


}
