<?php

namespace App\Http\Controllers;

use Illuminate\Http\Request;
use App\Http\Requests\AddEditMarkerRequest;
use App\Http\Requests\DownloadMarkerRequest;
use Illuminate\Support\Str;
use Illuminate\Support\Facades\Storage;
use App\Marker;
use App\Config;

use Endroid\QrCode\ErrorCorrectionLevel;
use Endroid\QrCode\LabelAlignment;
use Endroid\QrCode\QrCode;

use Dompdf\Dompdf;

class MarkerController extends Controller
{

    public function add(AddEditMarkerRequest $req)
    {
        $this->authorize('add', Marker::class);
        $marker = new Marker();
        
        // qrcode | barcode | pattern (not implemented)
        $markerType = $req->input('marker_type', Config::opt('APP_MARKER_TYPE'));

        // 3x3 | 3x3_PARITY65 | 4x4 (matrixCodeType)
        $matrixType = $req->input('matrix_type', Config::opt('APP_MARKER_MATRIX_TYPE')); 

        $markerName = Str::random(10).'.png';
        
        // $matrixType == '3x3' -> 00-63 (64 possibilities)
        // $matrixType == '3x3_PARITY65' -> 00-31 (32 possibilities)
        // $matrixType == '3x3_HAMMING63' -> 00-07 (8 possibilities)
        $markerValue = rand(0, 63);
        $markerImage = str_pad($markerValue, 2, '0', 0).'.png'; 

        // $matrixType == '4x4' -> 0000-8191 (8192 possibilities)
        if ($matrixType == '4x4') {
            $markerValue = rand(0, 8191);
            $markerImage = str_pad($markerValue, 4, '0', 0).'.png';
        }

        $markerPath = storage_path('app/barcodes/'.$matrixType.'/').$markerImage;

        //{"pattern_file":"filename.patt", "matrix_value": 8, "matrix_type":"3x3"}
        $props = [
            'marker_value' => $markerValue,
            'matrix_type' => $matrixType
        ];
        
        $barCodeSize = $req->input('marker_size', Config::opt('APP_MARKER_IMAGESIZE')); //1024

        if ($markerType == 'barcode') {
  
            // The file
            $filename = $markerPath;
            $destination = storage_path('app/public/markers/') . $markerName;

            // Get new dimensions
            list($width, $height) = getimagesize($filename);
            $new_width = $new_height = $barCodeSize;

            // Resample
            $image_p = imagecreatetruecolor($new_width, $new_height);
            $image = imagecreatefrompng($filename);
            imagecopyresampled($image_p, $image, 0, 0, 0, 0, $new_width, $new_height, $width, $height);

            // Save public disk
            imagepng($image_p, $destination);    

            $exists = Storage::exists("markers/$markerName");
            if(!$exists) {
                $contents = Storage::disk('public')->get("markers/$markerName");
                Storage::put("markers/$markerName", $contents);
                Storage::delete("markers/$markerName");
            }

        } elseif ($markerType == 'qrcode') {

            $sceneCode = $req->input('scene_code');
            $sceneURL = url("/ar/$sceneCode");

            $props['qrcode_value'] = $sceneURL;
    
            // Create a basic QR code
            $qrCode = new QrCode($sceneURL);
            $qrCode->setSize($barCodeSize);

            // Logo
            $logo_w = $logo_h = round(0.42 * $barCodeSize); // 30% of QrCode size
            $qrCode->setLogoPath($markerPath);
            $qrCode->setLogoSize($logo_w, $logo_h);
            
            // Caption
            $qrCodeCaptionEnabled = Config::opt('APP_QRCODE_CAPTION_ENABLED');
            $qrCodeCaptionEditable = Config::opt('APP_QRCODE_CAPTION_EDITABLE');
            if ($qrCodeCaptionEnabled) {
                if($qrCodeCaptionEditable) {
                    $qrCodeCaption = $req->input('qrcode_caption', Config::opt('APP_QRCODE_CAPTION_TEXT'));
                    $qrCodeFontSize = $req->input('qrcode_fontsize', Config::opt('APP_QRCODE_CAPTION_TEXT_FONTSIZE')); //(10% of QrCode size)
                } else {
                    $qrCodeCaption = Config::opt('APP_QRCODE_CAPTION_TEXT');
                    $qrCodeFontSize = Config::opt('APP_QRCODE_CAPTION_TEXT_FONTSIZE'); //(10% of QrCode size)
                }
                if(!empty($qrCodeCaption)) {
                    $qrCodeFont = base_path('vendor/endroid/qr-code/assets/fonts/noto_sans.otf');
                    $qrCode->setLabel($qrCodeCaption, $qrCodeFontSize, $qrCodeFont, LabelAlignment::CENTER);
                }
                //props
                $props['caption_text'] = $qrCodeCaption;
                $props['caption_fontsize'] = $qrCodeFontSize;
            }

            $qrCode->setErrorCorrectionLevel(new ErrorCorrectionLevel(ErrorCorrectionLevel::HIGH));
            $qrCode->setRoundBlockSize(true);
            $qrCode->setValidateResult(false);
            $qrCode->setWriterOptions(['exclude_xml_declaration' => true]);
    
            // Save it to a file
            //$qrCode->writeFile(storage_path('app/public/markers/') . $markerName);
            $contents = $qrCode->writeString();
            Storage::put("markers/$markerName", $contents);
        }
   
        $marker->scene_id = $req->input('scene_id');
        $marker->type = $markerType;
        $marker->props = json_encode($props);
        $marker->file = $markerName;
        $marker->save();

        $marker->props = $props;

        $marker->file = Storage::url("markers/$markerName");

        return response()->json(['response' => 'OK', 'marker' => $marker]);
    }



    public function edit(AddEditMarkerRequest $req, Marker $marker)
    {
  
        $this->authorize('edit', $marker);

        Storage::delete("markers/{$marker->file}");

        $marker_props = json_decode($marker->props, true);
        
        // qrcode | barcode | pattern (not implemented)
        $markerType = $marker->type;

        // 3x3 | 4x4 (matrixCodeType)
        $matrixType = $marker_props['matrix_type']; 

        $markerName = $marker->file;
        
        // $matrixType == '3x3' -> 00-63 (64 possibilities)
        // $matrixType == '4x4' -> 0000-8191 (8192 possibilities)
        $markerValue = $marker_props['marker_value']; 
        $markerImage = str_pad($markerValue, 2, '0', 0).'.png'; 

        $markerPath = storage_path('app/barcodes/'.$matrixType.'/').$markerImage;

        //{"pattern_file":"filename.patt", "matrix_value": 8, "matrix_type":"3x3"}
        $props = [
            'marker_value' => $markerValue,
            'matrix_type' => $matrixType
        ];
        
        $barCodeSize = $req->input('marker_size', Config::opt('APP_MARKER_IMAGESIZE')); //1024

        if ($markerType == 'barcode') {
  
            // The file
            $filename = $markerPath;
            $destination = storage_path('app/public/markers/') . $markerName;

            // Get new dimensions
            list($width, $height) = getimagesize($filename);
            $new_width = $new_height = $barCodeSize;

            // Resample
            $image_p = imagecreatetruecolor($new_width, $new_height);
            $image = imagecreatefrompng($filename);
            imagecopyresampled($image_p, $image, 0, 0, 0, 0, $new_width, $new_height, $width, $height);

            // Save public disk
            imagepng($image_p, $destination);    

            $exists = Storage::exists("markers/$markerName");
            if(!$exists) {
                $contents = Storage::disk('public')->get("markers/$markerName");
                Storage::put("markers/$markerName", $contents);
                Storage::delete("markers/$markerName");
            }

        } elseif ($markerType == 'qrcode') {

            $sceneCode = $req->input('scene_code');
            $sceneURL = url("/ar/$sceneCode");

            $props['qrcode_value'] = $sceneURL;
    
            // Create a basic QR code
            $qrCode = new QrCode($sceneURL);
            $qrCode->setSize($barCodeSize);

            // Logo
            $logo_w = $logo_h = round(0.42 * $barCodeSize); // 30% of QrCode size
            $qrCode->setLogoPath($markerPath);
            $qrCode->setLogoSize($logo_w, $logo_h);
            
            // Caption
            $qrCodeCaptionEnabled = Config::opt('APP_QRCODE_CAPTION_ENABLED');
            $qrCodeCaptionEditable = Config::opt('APP_QRCODE_CAPTION_EDITABLE');
            if ($qrCodeCaptionEnabled) {
                if($qrCodeCaptionEditable) {
                    $qrCodeCaption = $req->input('qrcode_caption', '');
                    $qrCodeFontSize = $req->input('qrcode_fontsize', 0.1 * $barCodeSize);
                } else {
                    $qrCodeCaption = Config::opt('APP_QRCODE_CAPTION_TEXT');
                    $qrCodeFontSize = Config::opt('APP_QRCODE_CAPTION_TEXT_FONTSIZE'); //(10% of QrCode size)
                }
                
                if(!empty($qrCodeCaption)) {
                    $qrCodeFont = base_path('vendor/endroid/qr-code/assets/fonts/noto_sans.otf');
                    $qrCode->setLabel($qrCodeCaption, $qrCodeFontSize, $qrCodeFont, LabelAlignment::CENTER);
                }
                //props
                $props['caption_text'] = $qrCodeCaption;
                $props['caption_fontsize'] = $qrCodeFontSize;
            }

            $qrCode->setErrorCorrectionLevel(new ErrorCorrectionLevel(ErrorCorrectionLevel::HIGH));
            $qrCode->setRoundBlockSize(true);
            $qrCode->setValidateResult(false);
            $qrCode->setWriterOptions(['exclude_xml_declaration' => true]);
    
            // Save it to a file
            //$qrCode->writeFile(storage_path('app/public/markers/') . $markerName);
            $contents = $qrCode->writeString();
            Storage::put("markers/$markerName", $contents);
        }
   
        $marker->scene_id = $req->input('scene_id');
        $marker->type = $markerType;
        $marker->props = json_encode($props);
        $marker->file = $markerName;
        $marker->save();

        $marker->props = $props;
        $marker->file = Storage::url("markers/$markerName");

        return response()->json(['response' => 'OK', 'marker' => $marker]);
    }



    public function download(DownloadMarkerRequest $req, Marker $marker)
    {
        $this->authorize('download', $marker);
        $file = "markers/{$marker->file}";
        $name = 'marker_' . date("Y_m_d_H_i_s");

        // PNG
        if($req->query('download_type') == 'png') {
            if($req->query('preview', '0') == '1') {
                return response(Storage::get($file))                            
                        ->header('Content-Type', 'image/png');
            }
            return Storage::download($file, $name);
        } else {
            // PDF
            $file_datauri = "data:image/png;base64,";
            $file_datauri .= base64_encode(Storage::get($file));

            $quantityOfMarkers = $req->query('quantity_markers', 1);
            $markerSize = $req->query('marker_size', 1);

            // markerSize => Columns, ImageWidth (px)
            $table = [[4, 170], [3, 230], [2, 270], [2, 350], [1, 400]];

            $cols = $table[$markerSize - 1][0];
            $lines = ceil($quantityOfMarkers/$cols);
            $imgWidth = $table[$markerSize - 1][1];
            
            $html =   "<!DOCTYPE html><html><head><title>$name</title>";
            $html .= '<link href="' . asset('css/custom.css') .'" rel="stylesheet">';
            $html .= '</head><body>';
            $html .=  '<table class="pdf-w714"><tbody>';

            for ($i=0; $i < $lines; $i++) {
                $html .= '<tr>';
                for ($j=0; $j < $cols; $j++) { 
                    if($quantityOfMarkers > 0) {
                        $html .= '<td class="pdf-td">';
                        $html .= '<img class="pdf-w'.$imgWidth.'" src="'.$file_datauri.'" />';
                        $html .= '</td>';
                    } else {
                        $html .= '<td></td>';
                    }
                    $quantityOfMarkers--;
                }
                $html .= '</tr>';
            }

            $html .= "</tbody></table></body></html>";

            $dompdf = new Dompdf();
            $dompdf->loadHtml($html);
            $dompdf->render(); // Render the HTML as PDF
            if($req->query('preview') == '1') {
                $output = $dompdf->output();                        // PDF content
                return response($output)                            // Displays the PDF in the browser
                        ->header('Content-Type', 'application/pdf');
            }
            return $dompdf->stream($name); // Download
        }
    }

}
