<?php

namespace App\Http\Controllers;

use Illuminate\Http\Request;
use App\Config;
use Illuminate\Support\Facades\Storage;
use Illuminate\Support\Str;
use Endroid\QrCode\QrCode;

class ConfigController extends Controller
{

    private $opts = [

        // THEME
        'APP_ICON_FILE',
        'APP_LOGO_FILE',
        'APP_THEME_COLOR',

        // SCENE
        'APP_SCENE_COMPONENT_LOADING_SCREEN_ENABLED',               
        'APP_SCENE_COMPONENT_LOADING_SCREEN_TITLE',                 
        'APP_SCENE_COMPONENT_LOADING_SCREEN_DOTSCOLOR',             
        'APP_SCENE_COMPONENT_LOADING_SCREEN_BACKGROUNDCOLOR',       

        // MARKER
        'APP_MARKER_TYPE',                                          // qrcode (default) | barcode | pattern
        'APP_MARKER_IMAGESIZE',                                     // 1024 (in px)
        'APP_MARKER_MATRIX_TYPE',                                   // 3x3 (default) | 4x4

        // QRCODE
        'APP_QRCODE_CAPTION_ENABLED',                               // 0 | 1
        'APP_QRCODE_CAPTION_EDITABLE',                              // 0 | 1
        'APP_QRCODE_CAPTION_TEXT',                                  // Caption text
        'APP_QRCODE_CAPTION_TEXT_FONTSIZE',                         // 102 (in px) 10% of APP_MARKER_IMAGESIZE

        // FILE UPLOAD
        'APP_UPLOAD_MAX_FILESIZE',                                  // 2048 (2MB in KB)
        'APP_UPLOAD_MAX_MEDIA_WIDTH',                               // 3840 (4K width in px)
        'APP_UPLOAD_MAX_MEDIA_HEIGHT',                              // 2160 (4K height in px)
        'APP_UPLOAD_MAX_MEDIA_DURATION',                            // 600  (600s - 10min)

    ];


    
    public function index(Request $request)
    {
 
        $this->authorize('index', Config::class);
        $arr_config = [];
        if($request->isMethod('post')) {
            $validation = [
                'APP_ICON_FILE' => '', // 4MB
                'APP_LOGO_FILE' => '', // 4MB
                'APP_THEME_COLOR' => '',
                'APP_SCENE_COMPONENT_LOADING_SCREEN_ENABLED' => 'required|numeric|in:0,1',              
                'APP_SCENE_COMPONENT_LOADING_SCREEN_TITLE' => 'sometimes|string|max:100',
                'APP_SCENE_COMPONENT_LOADING_SCREEN_DOTSCOLOR' => ['sometimes', 'regex:/^((0x){0,1}|#{0,1})([0-9A-F]{8}|[0-9A-F]{6}|[0-9A-F]{3})$/i'],
                'APP_SCENE_COMPONENT_LOADING_SCREEN_BACKGROUNDCOLOR' => ['sometimes', 'regex:/^((0x){0,1}|#{0,1})([0-9A-F]{8}|[0-9A-F]{6}|[0-9A-F]{3})$/i'],
                'APP_MARKER_TYPE' => 'required|in:qrcode,barcode',                      
                'APP_MARKER_IMAGESIZE' => 'required|numeric|min:100|max:5000', // 5000 x 5000px
                'APP_MARKER_MATRIX_TYPE' => 'required|string|in:3x3,4x4',
                'APP_QRCODE_CAPTION_ENABLED' => 'required|numeric|in:0,1',
                'APP_QRCODE_CAPTION_EDITABLE' => 'sometimes|numeric|in:0,1',
                'APP_QRCODE_CAPTION_TEXT' => '',              
                'APP_QRCODE_CAPTION_TEXT_FONTSIZE' => 'sometimes|numeric|min:10|max:500',     
                'APP_UPLOAD_MAX_FILESIZE' => 'required|numeric|min:512|max:102400', // 100MB
                'APP_UPLOAD_MAX_MEDIA_WIDTH' => 'required|numeric|min:800|max:5000',           
                'APP_UPLOAD_MAX_MEDIA_HEIGHT' => 'required|numeric|min:800|max:5000',          
                'APP_UPLOAD_MAX_MEDIA_DURATION' => 'required|numeric|min:10|max:600', // 10 min        
            ];

            if($request->hasFile('APP_ICON_FILE')) {
                $validation['APP_ICON_FILE'] = 'file|mimes:png,jpeg,gif|max:4096';
            }

            if($request->hasFile('APP_LOGO_FILE')) {
                $validation['APP_LOGO_FILE'] = 'file|mimes:png,jpeg,gif|max:4096';
            }

            if($request->input('APP_THEME_COLOR')) {
                $validation['APP_THEME_COLOR'] = ['present', 'regex:/^((0x){0,1}|#{0,1})([0-9A-F]{8}|[0-9A-F]{6}|[0-9A-F]{3})$/i'];
            }

            if($request->input('APP_QRCODE_CAPTION_TEXT')) {
                $validation['APP_QRCODE_CAPTION_TEXT'] = 'string|max:100';
            }    
            
            $request->validate($validation);

            // SAVE ICON FILE
            if($request->hasFile('APP_ICON_FILE') && $request->file('APP_ICON_FILE')->isValid()) {
                if(!empty(Config::opt('APP_ICON_FILE'))) {
                    Storage::disk('public')->delete(Config::opt('APP_ICON_FILE'));
                }
                $ext = $request->file('APP_ICON_FILE')->getClientOriginalExtension();
                $filename = 'icon.' . $ext;
                $request->file('APP_ICON_FILE')->storeAs('/', $filename, 'public');
                Config::opt('APP_ICON_FILE', $filename);
                $arr_config['ICON_FILE'] = Storage::disk('public')->url($filename);
            }

            // SAVE LOGO FILE
            if($request->hasFile('APP_LOGO_FILE') && $request->file('APP_LOGO_FILE')->isValid()) {
                if(!empty(Config::opt('APP_LOGO_FILE'))) {
                    Storage::disk('public')->delete(Config::opt('APP_LOGO_FILE'));
                }
                $ext = $request->file('APP_LOGO_FILE')->getClientOriginalExtension();
                $filename = 'logo.' . $ext;
                $request->file('APP_LOGO_FILE')->storeAs('/', $filename, 'public');
                Config::opt('APP_LOGO_FILE', $filename);
                $arr_config['LOGO_FILE'] = Storage::disk('public')->url($filename);
            }


            $tc = $request->input('APP_THEME_COLOR');
            $arr_config['THEME_COLOR'] = $tc? $tc: '#4e73df';

            // SAVE DYNAMIC CSS FILE
            $logo_file = Config::opt('APP_LOGO_FILE');
            $logo_file = empty($logo_file)? 'logo-holograma.png': $logo_file;
            $logo_url = Storage::disk('public')->url($logo_file);
            $css = <<<CSSF
                .dashboard_bg { 
                    background: url("{$logo_url}") center center no-repeat; 
                    background-size: 50%;
                }
                .holograma-btn, .holograma-btn:hover, .holograma-btn:focus {
                    color: #fff!important; 
                    background-color: {$arr_config['THEME_COLOR']}!important; 
                    border-color: {$arr_config['THEME_COLOR']}!important; 
                    box-shadow: none!important;
                }
                .holograma-color { 
                    color: {$arr_config['THEME_COLOR']}!important; 
                }
                .holograma-backgroundcolor { 
                    background: {$arr_config['THEME_COLOR']}!important; 
                }
CSSF;
            Storage::disk('public')->put('dynamic.css', $css);

            // UPDATE CONFIGS TABLE
            $options = $request->all();
            foreach ($options as $k => $v) {
                if(in_array($k, $this->opts) && $k != 'APP_ICON_FILE' && $k != 'APP_LOGO_FILE') {
                    $v = is_null($v) || empty($v)? '': $v;
                    Config::opt($k, $v);
                }
            }
            $request->session()->flash('status', __('Settings saved.'));
        }
        
        $configs = Config::all();
        foreach ($configs as $opt) {
            $arr_config[$opt->key] = $opt->value;
        }

        // Get upload_max_filesize in bytes
        $upLim = strtoupper(ini_get('upload_max_filesize'));
        $num = str_replace(['K', 'M', 'G'], '', $upLim);
        $end = substr($upLim, -1); //M K G
        switch ($end) {
            case 'K':
                $num *= 1024;
                break;
            case 'M':
                $num *= 1024**2;
                break;
            case 'G':
                $num *= 1024**3;
                break;
        }

        $arr_config['upload_max_filesize'] = $num/1024;
        
        return view('config.index', $arr_config);
    }



    public function previewQrCode(Request $req)
    {
        $this->authorize('preview_qrcode', Config::class);
        $caption = is_null($req->query('caption'))? '': $req->query('caption');
        $fontsize = (int) $req->query('fontsize', 15);

        // Create a basic QR code
        $qrCode = new QrCode("Holograma");
        $qrCode->setSize(1024);
        $qrCode->setLabel($caption, $fontsize);
        $contents = $qrCode->writeString();
        
        return response($contents)                            
                    ->header('Content-Type', 'image/png');
    }

}
